                                                                                /*
  OpenGL viewer of Molecular models
  by Andrew Ryzhkov and Arcady Antipin    (mailto:RedAndr@mail.ru)
                                                                                */
#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glaux.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <WinCon.h>

#define ver 3.78

#define sqr(a)  ((a)*(a))
#define at(a,x) atoms[bonds[i].a-1].x
#define len(x,y,z,xx,yy,zz) sqrt(sqr(x-xx)+sqr(y-yy)+sqr(z-zz))
#define x1 at(a,x)
#define x2 at(b,x)
#define y1 at(a,y)
#define y2 at(b,y)
#define z1 at(a,z)
#define z2 at(b,z)

#define MaxAtoms 1024

#define PI (3.14159265358979323846)
#define MAXsl 67
#define inca 5
#define radTheta 0.1
#define MAXSELECT 2560
        // max selected at once. four elements per object

        // all the global
struct atom { int type; GLdouble x,y,z,c;};
struct bond { int a,b;  GLdouble o;};
struct freq { int img;  GLdouble freqv,inten,x[MaxAtoms],y[MaxAtoms],z[MaxAtoms];};
struct grad { int znuc; GLdouble x,y,z;};

int NumAtoms, NumBonds, NumOrb,
    trad=0, bond=0, persp=0, wire=0, dline=0, chg=0, present=0, debug=0, morb=0,
    monum=1, atnum=0, spread=0, freqss=0, viewir=0, frcur=1, gradss=0, viewgr=0;

double BondTresh,MulPop[MaxAtoms][MaxAtoms];
    
struct atom atoms[  MaxAtoms];
struct bond bonds[  MaxAtoms];
struct freq freqs[3*MaxAtoms];
struct grad grads[  MaxAtoms];

GLfloat  bondmat[] = { 0.27, 0.57, 0.73, 0.5 };
GLfloat  irmat[]   = { 0.80, 0.57, 0.73, 0.5 };
GLdouble ax=0, ay=0, az=0, maxcor;
GLint    at_level=1000, bn_level=3000;
GLint    windW, windH;
GLint    windX=0, windY=0;
GLint    ap1=-1,ap2=-1,ap3=-1,ap4=-1;

GLuint selectBuf[MAXSELECT]; 
GLint   vp[4];

char    fname[256];

GLfloat mat_ambient1[] =    { 1.0,   1,   1, 1.0 },
        mat_specular1[] =   { 1.0, 1.0, 1.0, 1.0 },
        light_position1[]=  { 0, 0, 0, 0 },
        light_position2[]=  { 0, 0, 0, 0 },
        lm_ambient1[] =     { 0.2, 0.2, 0.2, 0.5 },
        light_color1[]=     { 1.0, 0.8, 0.7, 1 },
        light_color2[]=     { 0.7, 0.9, 1.0, 1 };

        // Define Atoms Colors
GLfloat at_def[68][4] = {
    { 0.6, 0.7, 0.8, 1.0 } /*H */,  { 0.1, 0.2, 0.5, 1.0 } /*He*/,
    { 0.7, 0.0, 0.7, 1.0 } /*Li*/,  { 0.5, 0.0, 0.3, 1.0 } /*Be*/,
    { 0.2, 0.2, 0.9, 1.0 } /*B */,  { 0.1, 0.1, 0.1, 1.0 } /*C */,
    { 0.2, 0.2, 0.9, 1.0 } /*N */,  { 0.9, 0.2, 0.2, 1.0 } /*O */,
    { 0.2, 0.9, 0.2, 1.0 } /*F */,  { 0.5, 0.0, 0.3, 1.0 } /*Ne*/,
    { 0.9, 0.0, 0.5, 1.0 } /*Na*/,  { 0.9, 0.2, 0.9, 1.0 } /*Mg*/,
    { 0.5, 0.2, 0.3, 1.0 } /*Al*/,  { 0.4, 0.3 ,0.1 ,1.0 } /*Si*/,
    { 1.0, 0.4, 0.4, 1.0 } /*P */,  { 0.9, 0.9, 0.3, 1.0 } /*S */,
    { 0.2, 0.9, 0.2, 1.0 } /*Cl*/,  { 0.5, 0.1, 0.3, 1.0 } /*Ar */,
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
    {0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},{0.96,0,0.88,1},
};

GLfloat PosChg[4]   = { 0.9, 0.1, 0.1, 1.0 };
GLfloat NegChg[4]   = { 0.1, 0.1, 0.9, 1.0 };
GLfloat MulPopMat[4]= { 0.8, 0.7, 0.2, 1.0 };
GLfloat at_rad[68] = {0.46,                              1.45,
                      1.51,1.12,0.97,0.77,0.71,0.65,0.71,1.56,
                      1.86,1.60,1.43,1.17,1.10,1.04,1.07,1.91,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2};

GLfloat at_radv[68]= {1.20,                              1.80,
                      2.50,2.10,1.90,1.70,1.50,1.40,1.35,2.00,
                      2.90,2.50,2.30,2.10,1.90,1.85,1.08,2.30,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2,
                      2,2,2,2,2,2,2,2,2,2};

GLfloat roma[4][4]/*[16]*/ = {
      1, 0, 0, 0,
      0, 1, 0, 0,
      0, 0, 1, 0,
      0, 0, 0, 1 };

GLfloat romb[4][4]/*[16]*/ = {
      1, 0, 0, 0,
      0, 1, 0, 0,
      0, 0, 1, 0,
      0, 0, 0, 1 };

GLdouble sinez[MAXsl];
GLdouble cosez[MAXsl];

#include "sphere.c"
#include "geomath.c"
#include "help.c"
#include "cr_bond.c"
#include "r_gamess.c"
#include "r_alc.c"
#include "r_xyz.c"
#include "reload.c"
#include "init.c"
#include "render.c"

void makeraster(GLenum mode, GLint x, GLint y) {
    if (windH == 0) windH = 1;
    if (mode == GL_SELECT) glGetIntegerv(GL_VIEWPORT, vp);
    glViewport (0, 0, windW, windH);
    glMatrixMode (GL_PROJECTION);
    glLoadIdentity();
    if (mode == GL_SELECT) gluPickMatrix(x,windH-y, 1, 1, vp);
    if (!persp) {
        if (windW <= windH)
                glOrtho (-maxcor, maxcor, -maxcor*windH/windW, maxcor*windH/windW, -maxcor, maxcor);
        else    glOrtho (-maxcor*windW/windH, maxcor*windW/windH, -maxcor, maxcor, -maxcor, maxcor);
    } else {
        gluPerspective (60.0, (GLdouble)windW / (GLdouble)windH, 1.0, 20.0);
    }
    glMatrixMode (GL_MODELVIEW);
    glLoadIdentity ();
    if (persp) glTranslated (0.0, 0.0, -2 * maxcor);  // viewing transform
}


void CALLBACK display(void)
{
        if (!spread) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glShadeModel(GL_SMOOTH);

        glPushMatrix();
        makeraster(GL_RENDER,0,0);
        Render (GL_RENDER);
        glPopMatrix();

        glFlush();
        auxSwapBuffers();
}

static GLint DoSelect(GLint x, GLint y)
{
        GLint hits,i,o;

        glSelectBuffer(MAXSELECT, selectBuf);
        (void)glRenderMode(GL_SELECT);
        glInitNames();
        glPushName((GLuint)~0);

        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glShadeModel(GL_SMOOTH);

        glPushMatrix();
        makeraster(GL_SELECT,x,y);
        Render(GL_SELECT);
        glPopMatrix();
    
        hits = glRenderMode(GL_RENDER); 

        if (hits <= 0)  return -1;
        else {
            if(debug) printf("----------\n* hits=%d\n* as:\n",hits);
            for(o=i=0;i<hits;i++) {
                if (selectBuf[o*4+1] > selectBuf[i*4+1]) o=i;
                if(debug) printf("%d %u %u %d\n",
                                selectBuf[i*4+0],
                                selectBuf[i*4+1],
                                selectBuf[i*4+2],
                                selectBuf[i*4+3]);
            }
            return selectBuf[o*4+3];
        }
}


void CALLBACK myReshape(GLsizei w, GLsizei h)
{
        windW = w;
        windH = h;
        makeraster(GL_RENDER,0,0);
        if (spread) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void printvec() {
  char img[2]=" I";
  printf("View IR vectors: %3d Frequency %9.2f%c Intensity: %9.5f\n",
  frcur,freqs[frcur].freqv,img[freqs[frcur].img],freqs[frcur].inten);
}

void printgrd() {
  double maxg,l;
  int   nmaxg,i;
  maxg=0;
  for(i=0;i<NumAtoms;i++){
    l = len(0,0,0,grads[i+1].x,grads[i+1].y,grads[i+1].z);
    if(l>maxg) {maxg=l; nmaxg=i;};
  }
  printf("Maximum gradient on atom %3d is dE/dx=%8.5f dE/dy=%8.5f dE/dz=%8.5f\n",
                  nmaxg+1, grads[nmaxg+1].x, grads[nmaxg+1].y, grads[nmaxg+1].z);
}

void CALLBACK dis1   (void) { maxcor/=1.2;   makeraster(GL_RENDER,0,0); }
void CALLBACK dis2   (void) { maxcor*=1.2;   makeraster(GL_RENDER,0,0); }
void CALLBACK dis3   (void) { persp=1-persp; makeraster(GL_RENDER,0,0); }
void CALLBACK dis5   (void) { wire=1-wire; }
void CALLBACK dis6   (void) { dline=1-dline; }
void CALLBACK dis7   (void) { chg=1-chg; if(debug){printf("Charge Mode: "); if(chg==1)puts("ON\n");else puts("OFF\n");} }
void CALLBACK dis8   (void) { morb=1-morb; printf("MO# %3d\n",monum);}
void CALLBACK dis9   (void) { viewir=1-viewir; if(viewir==1) printvec();}
void CALLBACK dis0   (void) { viewgr=1-viewgr; if(viewgr==1) printgrd();}
void CALLBACK moinc  (void) { if(monum!=NumOrb)monum++;else monum=1; printf("MO# %3d\n",monum);}
void CALLBACK modec  (void) { if(monum!=1)monum--;else monum=NumOrb; printf("MO# %3d\n",monum);}
void CALLBACK frinc  (void) { if(frcur!=NumAtoms*3)frcur++;else frcur=1; printvec(); }
void CALLBACK frdec  (void) { if(frcur!=1)frcur--;else frcur=NumAtoms*3; printvec(); }
void CALLBACK swrad  (void) { trad=1-trad; }
void CALLBACK swbond (void) { bond=1-bond; }
void CALLBACK move0  (void) { ax=0; ay=0; az=0; }
void CALLBACK movex1 (void) { rotxyz(1,0,0, inca,roma); }
void CALLBACK movex2 (void) { rotxyz(1,0,0,-inca,roma); }
void CALLBACK movey1 (void) { rotxyz(0,1,0, inca,roma); }
void CALLBACK movey2 (void) { rotxyz(0,1,0,-inca,roma); }
void CALLBACK movez1 (void) { rotxyz(0,0,1, inca,roma); }
void CALLBACK movez2 (void) { rotxyz(0,0,1,-inca,roma); }
void CALLBACK shatom (void) { atnum = 1 - atnum; }
void CALLBACK chklen (void) { printf("bond(%d,%d)=%lg\n",ap1+1,ap2+1,getBondLen(ap1,ap2)); }
void CALLBACK chkang (void) { printf("angle(%d,%d,%d)=%lg\n",ap1+1,ap2+1,ap3+1,getAngle(ap1,ap2,ap3)); }
void CALLBACK chkda  (void) { printf("dihedral(%d,%d,%d,%d)=%lg\n",ap1+1,ap2+1,ap3+1,ap4+1,getDA(ap1,ap2,ap3,ap4)); }
void CALLBACK chkzmat(void) { printf("%d   %d %lg   %d %lg   %d %lg\n",ap1+1,  ap2+1, getBondLen(ap1,ap2),
                                                                               ap3+1, getAngle(ap1,ap2,ap3),
                                                                               ap4+1, getDA(ap1,ap2,ap3,ap4) ); } 

#include "mouse.c"

/*-------------------------------------------------*\
|* Main Loop                                       *|
|* Open window with initial window size, title bar,*|
|* RGBA display mode, and handle input events.     *|
\*-------------------------------------------------*/
int main(int argc, char** argv)
{
  char  xstr[32];
  float mx[2],my[2],mz[2];
  int   i;

  SetConsoleTitle("ViewMol3D");

  printf("OpenGL viewer of Molecular models, ver. %4.2f\n"
         "Internet: http://RedAndr.tripod.com/vm3/\n\n",ver);

  if (argc < 2) {
    puts ("Usage : ViewMol3D <FileName.(mol/out/gms/mop/xyz)> [white] [debug] [allbonds] [atomic] [smear] [stick]\n");
    helpmsg ();
    return 1;
  }

  BondTresh = 0.5;

  for (i = 2; i < argc; i++) {
    if (strnicmp (argv[i], "white",    5) == 0) present   = 1;
    if (strnicmp (argv[i], "debug",    5) == 0) debug     = 1;
    if (strnicmp (argv[i], "allbonds", 5) == 0) BondTresh = 0;
    if (strnicmp (argv[i], "atomic",   6) == 0) atnum     = 1;
    if (strnicmp (argv[i], "smear",    5) == 0) spread = 1;
    if (strnicmp (argv[i], "stick",    5) == 0) { bond=1; dline =1; }
  }

  if (debug) printf ("BondTresh=%8.4f\n", BondTresh);

  strcpy (fname, argv[1]);
  strcpy (xstr, &fname[strlen (fname) - 3]);
  if ((strnicmp (xstr, "out", 3) == 0) ||
      (strnicmp (xstr, "gms", 3) == 0) ||
      (strnicmp (xstr, "log", 3) == 0) ||
      (strnicmp (xstr, "mop", 3) == 0)) {
    if (ReadMoleculeGAMESS (fname) != 0) {
      printf ("Can't open GAMESS/G98/MOPAC OUT file '%s'\n", fname);
      return -1;
    }
  } else if (strnicmp (xstr, "mol", 3) == 0) {
      if (ReadMoleculeALCHEMY (fname) != 0) {
        printf ("Can't open ALCHEMY MOL file '%s'\n", fname);
        return -1;
      } else dline=0;
  } else if (strnicmp (xstr, "xyz", 3) == 0) {
      if (ReadMoleculeXYZ (fname) != 0) {
        printf ("Can't open XYZ file '%s'\n", fname);
        return -1;
      } else dline=0;
  } else if (strnicmp (xstr, "dat", 3) == 0) {
      if (ReadMOPACdat(fname)!=0) {
      } else {
      }
  } else {
      puts ("I don't known that it's file. Please give me suitable extention (MOL/OUT/LOG/GMS/MOP)");
      return -2;
    }

  // Centering molecule to (0,0,0)
  mx[0]=mx[1]=atoms[0].x; my[0]=my[1]=atoms[0].y; mz[0]=mz[1]=atoms[0].z;
  for(i=1;i<NumAtoms;i++) {
    if(atoms[i].x>mx[0])mx[0]=atoms[i].x; else if(atoms[i].x<mx[1])mx[1]=atoms[i].x;
    if(atoms[i].y>my[0])my[0]=atoms[i].y; else if(atoms[i].y<my[1])my[1]=atoms[i].y;
    if(atoms[i].z>mz[0])mz[0]=atoms[i].z; else if(atoms[i].z<mz[1])mz[1]=atoms[i].z;
  }
  mx[0]+=(mx[1]-mx[0])/2; my[0]+=(my[1]-my[0])/2; mz[0]+=(mz[1]-mz[0])/2;
  for(i=0;i<NumAtoms;i++) { atoms[i].x-=mx[0]; atoms[i].y-=my[0]; atoms[i].z-=mz[0]; }

  auxInitDisplayMode (AUX_DOUBLE | AUX_RGBA | AUX_ACCUM | AUX_DEPTH16);
  windW = 460;
  windH = 440;
  auxInitPosition (128, 20, 128+windW, 20+windH);
  auxInitWindow (argv[1]);

  printf("OpenGL initialized:\n"
         "     Vendor  : %s\n"
         "     Renderer: %s\n"
         "     Version : %s\n",
         glGetString(GL_VENDOR),
         glGetString(GL_RENDERER),
         glGetString(GL_VERSION));
  helpmsg ();

  if (myinit () != 0) {
    puts ("Error OpenGL initialization.");
    return 2;
  };

  auxReshapeFunc (myReshape);
  auxKeyFunc   (AUX_1, dis1);
  auxKeyFunc   (AUX_2, dis2);
  auxKeyFunc   (AUX_3, dis3);
  auxKeyFunc   (AUX_4, swbond);
  auxKeyFunc   (AUX_5, dis5);
  auxKeyFunc   (AUX_6, dis6);
  auxKeyFunc   (AUX_7, dis7);
  auxKeyFunc   (AUX_8, dis8);
  auxKeyFunc   (AUX_9, dis9);
  auxKeyFunc   (AUX_0, dis0);
  auxKeyFunc   (AUX_RETURN, swrad);
  auxKeyFunc   (AUX_SPACE,  move0);
  auxKeyFunc   (AUX_UP,     movex1);
  auxKeyFunc   (AUX_DOWN,   movex2);
  auxKeyFunc   (AUX_LEFT,   movey1);
  auxKeyFunc   (AUX_RIGHT,  movey2);
  auxKeyFunc   (AUX_x,      movez1);
  auxKeyFunc   (AUX_z,      movez2);
  auxKeyFunc   (AUX_r,      Reload);
  auxKeyFunc   (AUX_v,      moinc);
  auxKeyFunc   (AUX_c,      modec);
  auxKeyFunc   (AUX_n,      shatom);
  auxKeyFunc   (AUX_l,      chklen);
  auxKeyFunc   (AUX_a,      chkang);
  auxKeyFunc   (AUX_d,      chkda);
  auxKeyFunc   (AUX_w,      frinc);
  auxKeyFunc   (AUX_q,      frdec);
  auxMouseFunc (AUX_RIGHTBUTTON, AUX_MOUSEDOWN, Mouse_right);
  auxMouseFunc (AUX_LEFTBUTTON , AUX_MOUSEDOWN, Mouse_left);
  auxMouseFunc (AUX_LEFTBUTTON , AUX_MOUSELOC,  Mouse_move);
  auxMouseFunc (AUX_LEFTBUTTON , AUX_MOUSEUP,   Mouse_leftup);
  
  auxMainLoop  (display);

  return 0;
}


int ReadMOPACdat(char* fname)
{
  char s[80];
  FILE *f;

printf("Open file:%s\n",fname);
  f=fopen(fname,"r");
  if(f==NULL) return -1;
  fgets(s,80,f);

  fclose(f);

if(debug)printf("Atoms:%3d Bonds:%3d\n",NumAtoms,NumBonds);
  return 0;
}